package console;

import java.util.Vector; 
import shared.*;

public class SystemConsole {
	
    private Vector<SystemConsoleEntry> m_consoleEntries;
    private Vector<Updatable> m_targets;
	private boolean m_autoScrollConsoleWindow = true;
	private int m_maxConsoleHistory = 512;

    public SystemConsole() {
    	m_consoleEntries = new Vector<SystemConsoleEntry>();
    	m_targets = new Vector<Updatable>();
    }
    
    // returns whether the console window should automatically scroll or not
    public boolean getAutoScrollConsoleWindow() {
    	return m_autoScrollConsoleWindow;
    }
    
    // returns the maximum number of lines the console can hold
    public int getMaxConsoleHistory() {
    	return m_maxConsoleHistory;
    }
    
    // change whether the console should automatically scroll or not
    public void setAutoScrollConsoleWindow(boolean autoScrollConsoleWindow) {
    	m_autoScrollConsoleWindow = autoScrollConsoleWindow;
    }
    
    // change the maximum number of lines the console can hold
    public void setMaxConsoleHistory(int maxConsoleHistory) {
    	if(maxConsoleHistory < 1) { return; }
    	m_maxConsoleHistory = maxConsoleHistory;
    }
    
    // adds a target which displays the contents of the console 
    public void addTarget(Updatable target) {
    	if(target == null) { return; }
    	
    	if(!m_targets.contains(target)) {
    		m_targets.add(target);
    	}
    }
    
    // removes the specified target
    public void removeTarget(Updatable target) {
    	if(target == null) { return; }
    	
    	m_targets.remove(target);
    }
    
    // removes all targets
    public void clearTargets() {
    	m_targets.clear();
    }
    
    // updates all targets so they reflect the new console data
    private void updateTargets() {
    	for(int i=0;i<m_targets.size();i++) {
    		m_targets.elementAt(i).update();
    	}
    }
    
    // returns the number of entries in the console
    public int size() {
    	return m_consoleEntries.size();
    }
    
    // returns a specific entry from the console
    public SystemConsoleEntry getConsoleEntry(int index) {
		if(index < 0 || index >= m_consoleEntries.size()) {
		    return null;
		}
		return m_consoleEntries.elementAt(index);
    }
    
    // writes a new entry to the console
    public void writeLine(String text) {
		m_consoleEntries.add(new SystemConsoleEntry(text));
		while(m_consoleEntries.size() > m_maxConsoleHistory) {
			m_consoleEntries.remove(0);
		}
		
		updateTargets();
    }
    
    // clears the console
    public void clear() {
    	m_consoleEntries.clear();
    	
    	updateTargets();
    }
    
    // writes the contents of the console to a string
    public String toString() {
    	String data = "";
    	SystemConsoleEntry e = null;
		for(int i=0;i<m_consoleEntries.size();i++) {
			e = m_consoleEntries.elementAt(i);
			data += e.getTimeString() + ": " + e.getText();
			if(i<m_consoleEntries.size() - 1) {
				data += "\n";
			}
		}
		return data;
    }
    
}
